<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Only allow POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit();
}

// Load configuration
require_once 'config.php';

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

// Validate required fields
$required = ['name', 'email', 'subject', 'message'];
$missing = [];
foreach ($required as $field) {
    if (empty($input[$field])) {
        $missing[] = $field;
    }
}

if (!empty($missing)) {
    http_response_code(400);
    echo json_encode(['error' => 'Missing required fields', 'fields' => $missing]);
    exit();
}

// Extract data
$name = htmlspecialchars($input['name']);
$email = filter_var($input['email'], FILTER_SANITIZE_EMAIL);
$phone = !empty($input['phone']) ? htmlspecialchars($input['phone']) : null;
$subject = htmlspecialchars($input['subject']);
$message = htmlspecialchars($input['message']);

// Validate email
if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid email address']);
    exit();
}

try {
    // Format date
    $currentDate = date('d/m/Y à H:i');
    
    // Generate email HTML
    $emailHtml = generateContactEmail($name, $email, $phone, $subject, $message, $currentDate);
    
    // Send email
    sendEmail(
        SMTP_HOST,
        SMTP_PORT,
        EMAIL_USER,
        EMAIL_PASS,
        EMAIL_USER,
        'info@taxiagadirairport.com',
        "💬 TaxiAgadirAirport - Contact: {$subject} - {$name}",
        $emailHtml,
        $email
    );
    
    // Success response
    http_response_code(200);
    echo json_encode([
        'success' => true,
        'message' => 'Message envoyé avec succès!'
    ]);
    
} catch (Exception $e) {
    error_log('Contact error: ' . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Erreur lors de l\'envoi du message',
        'details' => $e->getMessage()
    ]);
}

// Email generation function
function generateContactEmail($name, $email, $phone, $subject, $message, $currentDate) {
    $phoneRow = $phone ? "
    <div style='padding: 12px 0;'>
        <span style='font-weight: 700; color: #FB8C00; min-width: 140px; font-size: 14px;'>Téléphone:</span>
        <span style='color: #2c3e50; margin-left: 10px; word-break: break-word;'><a href='tel:{$phone}' style='color: #FB8C00; text-decoration: none; font-weight: 600;'>{$phone}</a></span>
    </div>
    " : '';
    
    // Escape message for HTML and preserve line breaks
    $messageHtml = nl2br(htmlspecialchars($message));
    
    return <<<HTML
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Nouveau Message de Contact</title>
</head>
<body style="margin: 0; padding: 0; font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; background-color: #f0f2f5; line-height: 1.6;">
    <div style="width: 100%; background-color: #f0f2f5; padding: 20px 0;">
        <div style="max-width: 600px; margin: 0 auto; background: white; border-radius: 16px; overflow: hidden; box-shadow: 0 10px 30px rgba(0,0,0,0.15);">
            <div style="background: linear-gradient(135deg, #FB8C00 0%, #F57C00 50%, #EF6C00 100%); padding: 40px 30px; text-align: center; position: relative;">
                <div style="position: absolute; top: 0; left: 0; right: 0; height: 4px; background: linear-gradient(90deg, #0D47A1, #1976D2, #0D47A1);"></div>
                <h1 style="color: white; margin: 0 0 8px 0; font-size: 32px; font-weight: 700; text-shadow: 0 2px 4px rgba(0,0,0,0.2);">💬 Taxi Agadir Airport</h1>
                <p style="color: #FFF3E0; margin: 0; font-size: 15px; font-weight: 500;">Nouveau Message de Contact</p>
                <div style="background: #1976D2; color: white; padding: 8px 20px; border-radius: 30px; display: inline-block; font-weight: bold; font-size: 11px; text-transform: uppercase; letter-spacing: 1px; margin-top: 15px; box-shadow: 0 4px 10px rgba(25, 118, 210, 0.3);">📧 NOUVEAU MESSAGE</div>
                <p style="color: #FFE0B2; font-size: 12px; margin-top: 10px; opacity: 0.9;">📅 Reçu le {$currentDate}</p>
            </div>
            
            <div style="padding: 35px 30px;">
                <div style="margin-bottom: 30px;">
                    <div style="background: linear-gradient(135deg, #FFF3E0 0%, #FFE0B2 100%); color: #E65100; font-size: 19px; font-weight: 700; margin-bottom: 18px; padding: 12px 15px; border-radius: 8px; border-left: 4px solid #0D47A1;">👤 Informations du Contact</div>
                    <div style="background: linear-gradient(135deg, #f8f9fa 0%, #ffffff 100%); border-radius: 12px; padding: 20px; border: 1px solid #e3e8ef;">
                        <div style="padding: 12px 0; border-bottom: 1px solid #e9ecef;">
                            <span style="font-weight: 600; color: #FB8C00; min-width: 140px; font-size: 14px;">Nom:</span>
                            <span style="color: #2c3e50; margin-left: 10px; word-break: break-word;"><strong>{$name}</strong></span>
                        </div>
                        <div style="padding: 12px 0; border-bottom: 1px solid #e9ecef;">
                            <span style="font-weight: 600; color: #FB8C00; min-width: 140px; font-size: 14px;">Email:</span>
                            <span style="color: #2c3e50; margin-left: 10px; word-break: break-word;"><a href="mailto:{$email}" style="color: #FB8C00; text-decoration: none; font-weight: 600;">{$email}</a></span>
                        </div>
                        {$phoneRow}
                    </div>
                </div>
                
                <div style="margin-bottom: 30px;">
                    <div style="background: linear-gradient(135deg, #FFF3E0 0%, #FFE0B2 100%); color: #E65100; font-size: 19px; font-weight: 700; margin-bottom: 18px; padding: 12px 15px; border-radius: 8px; border-left: 4px solid #0D47A1;">📝 Message</div>
                    <div style="background: white; border: 2px solid #E0E0E0; padding: 24px 26px; border-radius: 12px; box-shadow: 0 2px 8px rgba(0,0,0,0.08); position: relative;">
                        <div style="position: absolute; top: -15px; left: 20px; background: linear-gradient(135deg, #FB8C00, #F57C00); width: 35px; height: 35px; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-size: 18px; box-shadow: 0 3px 8px rgba(251, 140, 0, 0.3);">💬</div>
                        <div style="background: linear-gradient(135deg, #0D47A1 0%, #1976D2 100%); color: white; padding: 6px 16px; border-radius: 25px; font-size: 12px; font-weight: 700; text-transform: uppercase; letter-spacing: 0.5px; box-shadow: 0 3px 8px rgba(13, 71, 161, 0.3); display: inline-block; margin-bottom: 15px;">{$subject}</div>
                        <p style="margin: 0; color: #424242; line-height: 1.9; white-space: pre-wrap; font-size: 15px; padding-top: 5px;">{$messageHtml}</p>
                    </div>
                </div>
                
                <div style="background: linear-gradient(135deg, #FFF3E0 0%, #FFE0B2 100%); border: 2px solid #FB8C00; padding: 20px 24px; border-radius: 12px; margin-top: 30px; box-shadow: 0 4px 12px rgba(251, 140, 0, 0.15); position: relative;">
                    <div style="position: absolute; top: -15px; left: 20px; background: #FB8C00; width: 35px; height: 35px; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-size: 18px; box-shadow: 0 3px 8px rgba(251, 140, 0, 0.3);">⏰</div>
                    <p style="margin: 0; color: #E65100; font-size: 14px; font-weight: 600; line-height: 1.7;">
                        <strong style="font-size: 16px; display: block; margin-bottom: 8px; color: #BF360C; text-transform: uppercase; letter-spacing: 0.5px;">Action requise - RÉPONDRE RAPIDEMENT</strong>
                        Veuillez répondre au client dans les plus brefs délais pour assurer un service de qualité.
                    </p>
                </div>
            </div>
            
            <div style="background: linear-gradient(135deg, #263238 0%, #37474F 100%); color: white; padding: 30px 25px; text-align: center; font-size: 13px;">
                <div style="font-size: 18px; font-weight: 700; margin-bottom: 15px; color: #FB8C00;">🚕 Taxi Agadir Airport</div>
                <div style="height: 1px; background: linear-gradient(90deg, transparent 0%, #e0e0e0 50%, transparent 100%); margin: 20px 0;"></div>
                <p style="margin: 8px 0; opacity: 0.9;">📍 N˚ 07- Bloc A- Complexe TIVOLI - Secteur Balnéaire</p>
                <p style="margin: 8px 0; opacity: 0.9;">80010 - Agadir - Maroc</p>
                <p style="margin-top: 12px; opacity: 0.9;">📱 +212 608 003 030 | +212 661 444 189</p>
                <p style="margin: 8px 0; opacity: 0.9;">✉️ <a href="mailto:info@taxiagadirairport.com" style="color: #FB8C00; text-decoration: none; font-weight: 600;">info@taxiagadirairport.com</a></p>
                <p style="margin-top: 15px; font-size: 11px; opacity: 0.7;">Service Premium 24/7 - Transferts Aéroport & Navettes</p>
            </div>
        </div>
    </div>
</body>
</html>
HTML;
}

// Email sending function using native PHP mail
function sendEmail($host, $port, $username, $password, $from, $to, $subject, $html, $replyTo) {
    // Set headers
    $headers = "MIME-Version: 1.0\r\n";
    $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
    $headers .= "From: Taxi Agadir Airport <{$from}>\r\n";
    $headers .= "Reply-To: {$replyTo}\r\n";
    $headers .= "X-Mailer: PHP/" . phpversion();
    
    // Send email using native PHP mail
    $success = mail($to, $subject, $html, $headers);
    
    if (!$success) {
        error_log("Email failed to send to: {$to}");
        throw new Exception("Failed to send email");
    }
    
    error_log("Email sent successfully to: {$to}");
}
?>
